<?PHP

#
#   FILE:  SPT--Classification.php
#
#   METHODS PROVIDED:
#       Classification($ClassId, $Name = NULL, $TypeId = NULL)
#           - constructor
#       TypeName()
#           - get classification type as name
#       RecalcDepthAndFullName()
#           - rebuild classification full name and recalculate depth in hierarchy
#       RecalcResourceCount()
#           - recalculate number of resources assigned to class and any parent classes
#       ChildCount()
#           - calculate number of classifications that have this class as parent
#       Delete($DeleteParents = FALSE, $DeleteIfHasResources = FALSE, $DeleteIfHasChildren = FALSE)
#           - remove classification (and accompanying associations) from database
#       Id()
#       FullName()
#       Depth()
#       ParentId()
#       ResourceCount()
#           - get attributes
#       SegmentName($NewValue = DB_NOVALUE)
#       TypeId($NewValue = DB_NOVALUE)
#       LinkString($NewValue = DB_NOVALUE)
#       QualiferId($NewValue = DB_NOVALUE)
#           - get/set attributes
#
#   AUTHOR:  Edward Almasy
#
#   Part of the Scout Portal Toolkit
#   Copyright 2002-2003 Internet Scout Project
#   http://scout.wisc.edu
#

require_once(dirname(__FILE__)."/SPT--SPTDatabase.php");

# error status codes
define("CLASSSTAT_OK",                0);
define("CLASSSTAT_INVALIDID",         1);

class Classification {

    # ---- PUBLIC INTERFACE --------------------------------------------------

    # object constructor
    function Classification($ClassId, $Name = NULL, $TypeId = NULL)
    {
        # assume everything will turn out okay
        $this->ErrorStatus = CLASSSTAT_OK;
        
        # create DB handle for our use
        $this->DB =& new SPTDatabase();
        $DB =& $this->DB;

        # if class ID not given (indicating class must be created)
        if ($ClassId == NULL)
        {
            # parse classification name into separate segments
            $Segments = preg_split("/--/", $Name);

            # start out with parent as top
            $ParentId = -1;

            # for each segment
            foreach ($Segments as $Segment)
            {
                # look up classification with current parent and segment name
                $Segment = trim($Segment);
                $ClassId = $DB->Query("SELECT ClassificationId FROM Classifications "
                        ."WHERE ParentId = $ParentId AND SegmentName = '$Segment'",
                        "ClassificationId");

                # if classification not found
                if ($ClassId == NULL)
                {
                    # add new classification
                    $DB->Query("INSERT INTO Classifications "
                         ."(ClassificationTypeId, ParentId, SegmentName, ResourceCount) "
                         ."VALUES ($TypeId, $ParentId, '$Segment', 0)");

                    # set depth and full name for new classification
                    $ClassId = $DB->LastInsertId("Classifications");
                    $Class =& new Classification($ClassId);
                    $Class->RecalcDepthAndFullName();
                }

                # set parent to created or found class
                $ParentId = $ClassId;
            }
        }

        # save supplied or generated classification ID
        $this->Id = $ClassId;

        # load in attributes from database
        $DB->Query("SELECT * FROM Classifications"
                   ." WHERE ClassificationId = ".$ClassId);
        $this->DBFields = $DB->FetchRow();
        
        # set error status if class info not loaded
        if ($this->DBFields["ClassificationId"] != $this->Id)
        {
            $this->ErrorStatus = CLASSSTAT_INVALIDID;
        }
    }
    
    # check success of last call
    function Status() {  return $this->ErrorStatus;  }

    # get attributes
    function Id()            {  return $this->Id;  }
    function FullName()      {  return stripslashes($this->DBFields["ClassificationName"]);  }
    function Depth()         {  return $this->DBFields["Depth"];  }
    function ParentId()      {  return $this->DBFields["ParentId"];  }
    function ResourceCount() {  return $this->DBFields["ResourceCount"];  }

    # get/set attributes
    function SegmentName($NewValue = DB_NOVALUE) {  return stripslashes($this->UpdateValue("SegmentName", $NewValue));  }
    function TypeId($NewValue = DB_NOVALUE) {  return $this->UpdateValue("ClassificationTypeId", $NewValue);  }
    function LinkString($NewValue = DB_NOVALUE) {  return stripslashes($this->UpdateValue("LinkString", $NewValue));  }
    function QualifierId($NewValue = DB_NOVALUE) {  return $this->UpdateValue("QualifierId", $NewValue);  }

    # get classification type as name
    function TypeName()
    {
        # look up name in database and return value to caller
        return $this->DB->Query("SELECT ClassificationTypeName "
                    ."FROM ClassificationTypes "
                    ."WHERE ClassificationTypeId=".$this->DBFields["ClassificationTypeId"],
                "ClassificationTypeName");
    }

    # rebuild classification full name and recalculate depth in hierarchy
    function RecalcDepthAndFullName()
    {
        $DB =& $this->DB;

        # start with full classification name set to our segment name
        $FullClassName = $this->DBFields["SegmentName"];

        # assume to begin with that we're at the top of the hierarchy
        $Depth = 0;

        # while parent available
        $ParentId = $this->DBFields["ParentId"];
        while ($ParentId != -1)
        {
            # retrieve classification information
            $DB->Query("SELECT SegmentName, ParentId "
                    ."FROM Classifications "
                    ."WHERE ClassificationId=".$ParentId);
            $Record = $DB->FetchNextRowArray();

            # prepend segment name to full classification name
            $FullClassName = stripslashes($Record["SegmentName"])
                    ." -- ".$FullClassName;

            # increment depth value
            $Depth++;

            # move to parent of current classification
            $ParentId = $Record["ParentId"];
        }

        # for each child
        $DB->Query("SELECT ClassificationId "
                ."FROM Classifications "
                ."WHERE ParentId=".$this->Id);
        while ($Record = $DB->FetchNextRowArray())
        {
            # perform depth and name recalc
            $Child =& new Classification($Record["ClassificationId"]);
            $Child->RecalcDepthAndFullName();
        }

        # save new depth and full classification name
        $DB->Query("UPDATE Classifications SET "
                ."Depth=".$Depth.", "
                ."ClassificationName='".addslashes($FullClassName)."' "
                ."WHERE ClassificationId=".$this->Id);
        $this->DBFields["ClassificationName"] = $FullClassName;
        $this->DBFields["Depth"] = $Depth;
    }

    # recalculate number of resources assigned to class and any parent classes
    function RecalcResourceCount()
    {
        $DB =& $this->DB;

        # retrieve new count of resources directly associated with class
        $DB->Query("SELECT COUNT(*) AS ResourceCount "
                ."FROM ResourceClassInts "
                ."WHERE ClassificationId=".$this->Id);
        $Record = $DB->FetchNextRowArray();
        $this->DBFields["ResourceCount"] = $Record["ResourceCount"];

        # add on resources associated with all children
        $this->DBFields["ResourceCount"] += $DB->Query(
                "SELECT SUM(ResourceCount) AS ResourceCountTotal "
                    ."FROM Classifications "
                    ."WHERE ParentId = ".$this->Id,
                "ResourceCountTotal");

        # save new resource count
        $DB->Query("UPDATE Classifications SET "
                ."ResourceCount=".$this->DBFields["ResourceCount"]." "
                ."WHERE ClassificationId=".$this->Id);

        # update resource count for our parent (if any)
        if ($this->DBFields["ParentId"] != -1)
        {
            $Class =& new Classification($this->DBFields["ParentId"]);
            $Class->RecalcResourceCount();
        }
    }

    # calculate number of classifications that have this class as parent
    function ChildCount()
    {
        # return count of classifications that have this one as parent
        return $this->DB->Query("SELECT COUNT(*) AS ClassCount "
                    ."FROM Classifications "
                    ."WHERE ParentId=".$this->Id,
                "ClassCount");
    }

    # remove classification (and accompanying associations) from database
    function Delete($DeleteParents = FALSE, 
            $DeleteIfHasResources = FALSE, $DeleteIfHasChildren = FALSE)
    {
        $DB =& $this->DB;

        # if no resources or okay to delete with resources
        #         and no children or okay to delete with children
        if (($DeleteIfHasResources || ($this->ResourceCount() == 0))
                && ($DeleteIfHasChildren || ($this->ChildCount() == 0)))
        {
            # delete this classification
            $DB->Query("DELETE FROM Classifications "
                    ."WHERE ClassificationId=".$this->Id);
            if ($DeleteIfHasResources)
            {
                $DB->Query("DELETE FROM ResourceClassInts "
                        ."WHERE ClassificationId=".$this->Id);
            }

            # delete parent classification (if requested)
            if (($DeleteParents) && ($this->DBFields["ParentId"] != -1))
            {
                $Parent =& new Classification($this->DBFields["ParentId"]);
                $Parent->Delete(
                        TRUE, $DeleteIfHasResources, $DeleteIfHasChildren);
            }
        }
    }


    # ---- PRIVATE INTERFACE -------------------------------------------------

    var $DB;
    var $DBFields;
    var $Id;
    var $ErrorStatus;

    # convenience function to supply parameters to Database->UpdateValue()
    function UpdateValue($FieldName, $NewValue)
    {
        return $this->DB->UpdateValue("Classifications", $FieldName, $NewValue,
                               "ClassificationId = ".$this->Id,
                               $this->DBFields, TRUE);
    }
}


?>
